const { adGroupsListSP, adGroupsListSB } = require("./service/groups");
const { campaignsListSB, campaignsListSP } = require("./service/campaign");
const { portfoliosList } = require("./service/portafolio");
const { productsAdsSP } = require("./service/asins");
const amazonAdsConfig = require("./config/amazon");
const express = require("express");
const axios = require("axios");
const _ = require("lodash");
const cors = require("cors");

require("dotenv").config();

const db = require('./config/db');

const app = express();
const PORT = process.env.PORT || 8080;

let lastResponse = {};

app.use(cors());

app.get("/", async (req, res) => {
    console.log("Cliente conectado");
    try {
        const [rows, fields] = await db.query("SELECT 1 + 1 AS solution");
        console.log("Resultado de la consulta de prueba:", rows[0].solution);
        res.status(200).json({ message: "Ok", db_status: "Conectado", test_query_result: rows[0].solution });
    } catch (error) {
        console.error("Error en la consulta de prueba a la BD:", error);
        res.status(500).json({ message: "Error", db_status: "Error en la conexión" });
    }
});

function calculateBranchPointsRecursive(currentUserId, rootCalculatorId, isRootQualified, childrenMap, usersMap, pointsMap) {
    const userNode = childrenMap.get(currentUserId);
    if (!userNode) {
        return 0;
    }

    let branchPoints = 0;
    const children = [...userNode.left, ...userNode.right];

    for (const childId of children) {
        const childUser = usersMap.get(childId);
        if (!childUser) continue;

        let pointsFromThisChild = 0;

        if (isRootQualified || childUser.id_user_sponsor === rootCalculatorId) {
            pointsFromThisChild = pointsMap.get(childUser.id_account_type) || 0;
        }
        
        branchPoints += pointsFromThisChild;
        
        branchPoints += calculateBranchPointsRecursive(childId, rootCalculatorId, isRootQualified, childrenMap, usersMap, pointsMap);
    }

    return branchPoints;
}


app.get("/binaryCut", async (req, res) => {
    console.log("Procesando /binaryCut (versión con lógica de patrocinio)...");
    try {
        const [allUsers] = await db.query(`
            SELECT 
                u.id, u.name, u.last_name, u.expiration_membership_date, u.id_account_type,
                c.id_user_sponsor 
            FROM users u
            LEFT JOIN classified c ON u.id = c.user_id
        `);
        const [allAccountTypes] = await db.query("SELECT account_type_id as id, points FROM account_type_points_money");
        const [allClassifiedData] = await db.query("SELECT user_id, position, user_above FROM classified");

        const now = new Date();

        const usersMap = new Map(allUsers.map(u => [u.id, u]));
        const pointsMap = new Map(allAccountTypes.map(at => [at.id, Number(at.points)]));

        const childrenMap = new Map();
        for (const row of allClassifiedData) {
            const parentId = Number(row.user_above);
            if (!parentId) continue;

            if (!childrenMap.has(parentId)) {
                childrenMap.set(parentId, { left: [], right: [] });
            }

            if (row.position == 0) {
                childrenMap.get(parentId).left.push(row.user_id);
            } else if (row.position == 1) {
                childrenMap.get(parentId).right.push(row.user_id);
            }
        }

        const results = [];
        for (const user of allUsers) {
            const expirationDate = user.expiration_membership_date ? new Date(user.expiration_membership_date) : null;
            const isActive = expirationDate && expirationDate > now;

            if (!isActive) {
                results.push({ userId: user.id, name: `${user.name} ${user.last_name}`, totalPoints: 0, status: "inactivo", details: {} });
                continue;
            }

            const directChildrenNode = childrenMap.get(user.id) || { left: [], right: [] };
            
            const hasDirectLeft = directChildrenNode.left.some(childId => usersMap.get(childId)?.id_user_sponsor === user.id);
            const hasDirectRight = directChildrenNode.right.some(childId => usersMap.get(childId)?.id_user_sponsor === user.id);
            const isQualified = hasDirectLeft && hasDirectRight;

            let totalLeftPoints = 0;
            let totalRightPoints = 0;

            for (const leftChildId of directChildrenNode.left) {
                const leftChildUser = usersMap.get(leftChildId);
                if(leftChildUser) {
                    if (isQualified || leftChildUser.id_user_sponsor === user.id) {
                        totalLeftPoints += pointsMap.get(leftChildUser.id_account_type) || 0;
                    }
                    totalLeftPoints += calculateBranchPointsRecursive(leftChildId, user.id, isQualified, childrenMap, usersMap, pointsMap);
                }
            }

            for (const rightChildId of directChildrenNode.right) {
                const rightChildUser = usersMap.get(rightChildId);
                if(rightChildUser){
                    if (isQualified || rightChildUser.id_user_sponsor === user.id) {
                        totalRightPoints += pointsMap.get(rightChildUser.id_account_type) || 0;
                    }
                    totalRightPoints += calculateBranchPointsRecursive(rightChildId, user.id, isQualified, childrenMap, usersMap, pointsMap);
                }
            }
            
            const totalPoints = isQualified ? (totalLeftPoints + totalRightPoints) : 0;

            let directPoints = 0;
            [...directChildrenNode.left, ...directChildrenNode.right].forEach(childId => {
                const child = usersMap.get(childId);
                if (child && child.id_user_sponsor === user.id) {
                    directPoints += pointsMap.get(child.id_account_type) || 0;
                }
            });

            results.push({
                userId: user.id,
                name: `${user.name} ${user.last_name}`,
                totalPoints: totalPoints,
                status: isQualified ? "activo_y_calificado" : "activo_no_calificado",
                details: {
                    puntosRamaIzquierda: totalLeftPoints,
                    puntosRamaDerecha: totalRightPoints,
                    puntosSoloDeDirectosPatrocinados: directPoints,
                }
            });
        }

        res.status(200).json(results);

    } catch (error) {
        console.error("❌ Error en /binaryCut:", error);
        res.status(500).json({ message: "Error procesando el corte binario", error: error.message });
    }
});

app.listen(PORT, () => console.log(`BOT escuchando en el puerto ${PORT}...`));

const getProfilesID = async () => {
    const apiUrl = "https://advertising-api.amazon.com/adsAccounts/list";
    const accessToken = await amazonAdsConfig.getValidAccessToken();
    const headers = {
        "Authorization": `Bearer ${accessToken}`,
        "Amazon-Advertising-API-ClientId": amazonAdsConfig.clientId,
    };

    try {
        const response = await axios.post(apiUrl, null, { headers });
        return { data: response.data }
    } catch (error) {
        console.log("Error fetching profile IDs:", error.message);
        return null;
    }
}

const formatearUsuarios = async () => {
    const response = await getProfilesID();
    if (!response || !response.data || !response.data.adsAccounts) {
        console.log("FormatearUsuarios: No accounts data received.");
        return [];
    }
    const result = response.data.adsAccounts
        .filter(account => account.countryCodes && account.countryCodes.includes("US"))
        .map(account => {
            const alternateIdUS = account.alternateIds ? account.alternateIds.find(
                alt => alt.countryCode === "US" && alt.profileId
            ) : undefined;
            return {
                ...account,
                alternateIds: alternateIdUS ? [alternateIdUS] : []
            };
        });

    const finalResult = result
        .map(account => {
            const profileId = account.alternateIds && account.alternateIds[0] ? account.alternateIds[0].profileId : null;
            if (!profileId) {
                console.warn("Cuenta sin profileId US válido:", account.accountName, "ID:", account.accountId);
                return null;
            }
            return {
                label: account.accountName,
                value: profileId.toString()
            };
        })
        .filter(item => item && item.value);

    return finalResult;
}

function mergeCampaignsWithASIN(campaigns, productos) {
    return productos.map(producto => ({
        ...producto,
        campaigns: campaigns.filter(campaign => campaign.campaignId === producto.campaignId),
    }));
}

function mergeCampaignsWithPortafolios(portafolios, campaigns) {
    return portafolios.map(portafolio => ({
        ...portafolio,
        campaigns: campaigns.filter(campaign => campaign.portfolioId === portafolio.portfolioId),
    }));
}

function mergeCampaignsWithAdGroups(campaigns, adGroups, type) {
    return campaigns.map(campaign => ({
        ...campaign,
        type: type,
        adGroups: adGroups.filter(adGroup => adGroup.campaignId === campaign.campaignId),
    }));
}

async function getData(accountId) {
    const [adGroupsDataSP, adGroupsDataSB, campaignsDataSP, campaignsDataSB, portafoliosData, productosData] = await Promise.all([
        adGroupsListSP(accountId),
        adGroupsListSB(accountId),
        campaignsListSP(accountId),
        campaignsListSB(accountId),
        portfoliosList(accountId),
        productsAdsSP(accountId),
    ]);

    const campaignsMergeSP = mergeCampaignsWithAdGroups(campaignsDataSP.campaigns, adGroupsDataSP.adGroups, "SP");
    const campaignsMergeSB = mergeCampaignsWithAdGroups(campaignsDataSB.campaigns, adGroupsDataSB.adGroups, "SB");

    const campaignsData = [...campaignsMergeSB, ...campaignsMergeSP]
    const adGroupsData = [...adGroupsDataSP.adGroups, ...adGroupsDataSB.adGroups]

    const portafoliosMerge = mergeCampaignsWithPortafolios(portafoliosData.portfolios, campaignsData)
    const productosMerge = mergeCampaignsWithASIN(campaignsData, productosData.productAds)

    return {
        campaigns: campaignsData,
        adgroups: adGroupsData,
        portfolios: portafoliosMerge,
        productos: productosMerge
    }
}

function findDifferences(obj1, obj2) {
    return {
        campaigns: _.differenceWith(obj1.campaigns, obj2.campaigns, _.isEqual),
        adgroups: _.differenceWith(obj1.adgroups, obj2.adgroups, _.isEqual),
        portfolios: _.differenceWith(obj1.portfolios, obj2.portfolios, _.isEqual)
    };
}

async function checkForChanges() {
    try {
        console.log("Ejecutando...");

        const usuarios = await formatearUsuarios();
        const chunkSize = 40;

        const chunkArray = (array, size) => {
            const result = [];
            for (let i = 0; i < array.length; i += size) {
                result.push(array.slice(i, i + size));
            }
            return result;
        };

        const usuariosChunks = chunkArray(usuarios, chunkSize);

        for (const chunk of usuariosChunks) {
            const changesPromises = chunk.map(async ({ value: accountId }) => {
                try {
                    const axiosInstance = await amazonAdsConfig.createAxiosInstance(accountId);
                    await axiosInstance.get("/v2/profiles");

                    if (!lastResponse.hasOwnProperty(accountId)) {
                        lastResponse[accountId] = {};
                    }

                    const response = await getData(accountId);

                    if (Object.keys(lastResponse[accountId]).length !== 0) {
                        const differences = findDifferences(response, lastResponse[accountId]);
                        if (
                            differences.campaigns.length ||
                            differences.adgroups.length ||
                            differences.portfolios.length
                        ) {
                            console.log("🔍 Diferencias detectadas para", accountId, ":", JSON.stringify(differences, null, 2));
                            console.log("🔔 Cambio detectado, notificando al servidor...");
                            await notifyServer(accountId);
                        }
                    }

                    lastResponse[accountId] = response;
                } catch (err) {
                    console.error("❌ Error en cuenta", accountId, err.message);
                }
            });

            await Promise.all(changesPromises);
        }
    } catch (error) {
        console.error("❌ Error al obtener usuarios:", error.message);
    }
}

async function notifyServer(accountID) {
    try {
        await axios.post(process.env.SERVER_URL, {
            mensaje: "Cambio detectado en la API",
            accountId: accountID
        });
        console.log("✅ Notificación enviada.");
    } catch (error) {
        console.error("⚠️ Error al notificar al servidor:", error.message);
    }
}

async function runCheckLoop() {
    try {
        await checkForChanges();
    } catch (e) {
        console.error("❌ Error en runCheckLoop:", e.message);
    } finally {
        setTimeout(runCheckLoop, 5000);
    }
}

